#ifndef AHLGREN_WAM_PROGRAM
#define AHLGREN_WAM_PROGRAM

#include <vector>
#include <queue>
#include <deque>
#include <map>
#include <cmath>
#include <algorithm>
#include <unordered_set>
#include <chrono>

#include "global.h"
#include "functor.h"
#include "wam.h"
#include "parameters.h"
#include "statistics.h"
#include "graph.h"
#include "mode.h"
#include "levenshtein.h"
#include "unification.h"


namespace lp {
	// Exceptions
	class induction_exception : public std::exception {};
	// Thrown and caught inside generalize()
	class search_completed : public induction_exception {};
	class search_aborted : public induction_exception {};
	class bottom_rejected : public induction_exception {};
	// Cancel not only current search lattice, but the entire induction
	class induction_aborted : public induction_exception {};
	// Thrown and caught outside generalize()
	class builtin_overwrite : public induction_exception {};

	class Termindex;

	class Instances {
	public:
		typedef std::set<Functor> set_t;
		typedef set_t::iterator iterator;
		typedef set_t::const_iterator const_iterator;
		typedef std::vector<Functor> vec_t;
		typedef vec_t::iterator viter;
		typedef vec_t::const_iterator vciter;
		Instances() {}
		~Instances() { dealloc(); }
		void dealloc() { clear(); }
		void clear() { insts.clear(); instv.clear(); }
		bool empty() const { assert(insts.size() == instv.size()); return instv.empty(); }
		std::size_t size() const { assert(insts.size() == instv.size()); return instv.size(); }
		void add(const Functor& t) { 
			assert( t.is_ground() ); 
			auto p = insts.insert(t);
			if (p.second) instv.push_back(t);
			assert(insts.size() == instv.size());
		}
		void add(Instances& ic) { 
			for (auto& i : ic) {
				if (insts.find(i) == insts.end()) {
					insts.insert(i);
					instv.push_back(i);
				} else {
					// already in insts
				}
			}
			ic.clear(); // no dealloc on ic
		}
		// Find
		iterator find(const Functor& t) { return insts.find(t); }
		const_iterator find(const Functor& t) const { return insts.find(t); }
		iterator find_end() { return insts.end(); }
		const_iterator find_end() const { return insts.end(); }
		// Iterators
		viter begin() { return instv.begin(); }
		viter end() { return instv.end(); }
		vciter begin() const { return instv.begin(); }
		vciter end() const { return instv.end(); }
	protected:
		std::set<Functor> insts;
		std::vector<Functor> instv;
	};

	typedef std::map<Functor,id_type> term2var_t;
	typedef std::multimap<sign_t,std::pair<Functor::seq_const_iterator,int>> literal_map;
	struct implication {
		implication() : is_dynamic(false) {}
		// No antecedent, only Consequent
		implication(const Functor& f, bool d) : ante(), cons(f), is_dynamic(d) {}
		implication(Functor&& f, bool d) : ante(), cons(std::move(f)), is_dynamic(d) {}
		// Antecedent and Consequent
		implication(const std::vector<Functor>& v, const Functor& f, bool d) : ante(v), cons(f), is_dynamic(d) {}
		implication(std::vector<Functor>&& v, Functor&& f, bool d) : ante(std::move(v)), cons(std::move(f)), is_dynamic(d) {}
		implication(Functor&& a, Functor&& f, bool d) : ante(1,std::move(a)), cons(std::move(f)), is_dynamic(d) {}
		implication(const Functor& a, const Functor& f, bool d) : ante(1,a), cons(f), is_dynamic(d) {}

		void clear() { ante.clear(); cons.clear(); is_dynamic = false; }
		std::vector<Functor> ante;
		Functor cons;
		bool is_dynamic;
	};
	typedef std::vector<implication> implications;

	struct Constraints {
		std::multimap<Mode,Functor::position> dependent_vars;
		// Unconditional Constraints
		std::map<Functor,implication> uncond;
		// Conditional Constraints
		std::map<Functor,implication> cond;
	};


	class Program {
	public:
		typedef lp::concept concept;
		typedef lp::clause clause;
		typedef lp::sign_t sign_t;
		typedef std::pair<sign_t,clause> cl_def;
		typedef lp::code::iterator concept_iterator;
		typedef lp::code::const_iterator concept_const_iterator;
		typedef std::vector<Mode> modev_type;
		typedef modev_type::iterator mode_iterator;
		typedef modev_type::const_iterator mode_const_iterator;
		// Determinations
		typedef std::multimap<sign_t,sign_t> det_type;

		Program() { }
		Program(Program&& p) { *this = std::move(p); }
		Program& operator=(Program&& p) { 
			if (this != &p) { 
				kb = std::move(p.kb); 
				modev = std::move(p.modev); 
				params = std::move(p.params); 
				includes = std::move(p.includes); 
				excludes = std::move(p.excludes);
				imp = std::move(p.imp); 
				block = std::move(p.block);
			}
			return *this; 
		}

		// Clear KB
		void clear_kb() { kb.clear(); modev.clear(); includes.clear(); excludes.clear(); imp.clear(); block.clear(); }
		void clear() { clear_kb(); params = parameters(); }

		// Queries
		bool query(const Functor& q, deadline_t* deadline = nullptr); // called with once
		bool query(const clause&, long long* ic = nullptr, deadline_t* deadline = nullptr); // once
		//bool query(clause&&); // once
		bool query(const Functor&, lp::qconstraints, deadline_t* deadline = nullptr); // once
		int query(const Functor&, const std::set<id_type>& qvars, deadline_t* deadline = nullptr);
		int query(const clause&, const std::set<id_type>& qvars, deadline_t* deadline = nullptr);
		//int query(clause&&, const std::set<id_type>& qvars, deadline_t* deadline = nullptr);
		int query(const Functor&, lp::qconstraints&, lp::answer_type&, deadline_t* deadline = nullptr);
		int query(const clause&, lp::qconstraints&, lp::answer_type&, long long* ic = nullptr, deadline_t* deadline = nullptr);
		//int query(clause&&, lp::qconstraints&, lp::answer_type&, deadline_t* deadline = nullptr);
		int query(const Functor&, lp::qconstraints&, std::vector<Functor>&, deadline_t* deadline = nullptr); // build
		// Execute instructions, return number of successes
		int exec(const lp::qconstraints& qc, answer_type& ans, const clause* external_query, long long* instr_count = nullptr, deadline_t* deadline = nullptr);

		// Is f of type?
		bool check_type(const Functor& f, id_type type);

		// Iterate through Mode Declarations
		mode_const_iterator mode_begin() const { return modev.begin(); }
		mode_const_iterator mode_end() const { return modev.end(); }
		mode_iterator mode_begin() { return modev.begin(); }
		mode_iterator mode_end() { return modev.end(); }
		// Iterate through concepts
		concept_iterator con_begin() { return kb.begin(); }
		concept_iterator con_end() { return kb.end(); }
		concept_const_iterator con_begin() const { return kb.begin(); }
		concept_const_iterator con_end() const { return kb.end(); }
		concept_iterator erase_concept(concept_iterator);
		bool erase_concept(const sign_t& s);
		// Iterate through all clauses of a concept
		typedef lp::concept::iterator cl_iterator;
		typedef lp::concept::const_iterator cl_const_iterator;
		std::pair<cl_iterator,cl_iterator> concept_range(const lp::sign_t&);
		std::pair<cl_const_iterator,cl_const_iterator> concept_range(const lp::sign_t&) const;
		void erase_clause(concept_iterator, cl_iterator);
		void erase_clause(const lp::sign_t&, cl_iterator);
		// Iterate through all instructions of a clause
		typedef lp::clause::iterator instr_iterator;
		typedef lp::clause::const_iterator instr_const_iterator;
		// Make WAM instructions in KB
		void compile_insert(const Functor& cl, const std::set<id_type>& qvars, bool collect, bool at_end = true);
		inline void compile_insert(const Functor& cl, bool c, bool e = true) { std::set<id_type> m; return compile_insert(cl,m,c,e); }
		// Set WAM state in beginning of execution mode
		wam_state wam_reset(const clause* extern_query_ptr);
		// Restore WAM state to ws
		void wam_reset(const wam_state& ws);
		// Easy Inserts
		void push_front(const Functor&);
		void push_front(const cl_def&);
		void push_back(const Functor&);
		void push_back(const cl_def&);
		void push_back(const clause&); // queries only
		void pop_back(const sign_t&);

		// Covering
		bool covers(const clause& example, long long* ic = nullptr, double* time = nullptr, deadline_t* deadline = nullptr);
		bool covers(const Functor& candidate, const Functor& example, deadline_t* deadline);
		bool covers(const Functor& candidate, const clause& example, deadline_t* deadline);
		//template <class Iter> int covers_terms(const Functor& candidate, Iter beg, Iter end);
		template <class Iter> int covers_clauses(const Functor& candidate, Iter beg, Iter end, deadline_t* deadline);

		// ILP generalization methods
		// Generalize using method
		std::vector<std::pair<sign_t,stat>> generalize(const char*);
		//stat generalize(id_type method);
		// Use NrSample
		stat generalize_nrsample(const sign_t&);
		// Iterated Deepening version of NrSample
		stat generalize_id_nrsample(const sign_t&);
		// Use Emulated version of NrSample (without constraints)
		stat generalize_emulated_nrsample(const sign_t&);
		// Use Enumeration (Aleph's default)
		stat generalize_enumerate(const sign_t&);
		// Use best-first search (Progol A*)
		stat generalize_best_first(const sign_t&);

		// Extract Examples (according to modeh)
		template <typename PI, typename NI> // output iterators
		void examples_to_functors(const sign_t& sign, PI pi, NI ni, bool strip_negatives = false);
		// Extract Examples (according to modeh) and convert them into queries
		template <typename PI, typename NI> // output iterators
		void examples_to_queries(const sign_t&, PI pi, NI ni, bool strip_negatives = false);

		// Detect Properties before induction
		// Detect dependent variables
		void detect_dependent(const std::list<Functor>& posv, const std::list<Functor>& negv,
			const std::map<const Functor*,const clause*>& fun2cl, Constraints& constr);
		// Detect Reflexivity
		void detect_reflexivity(const std::list<Functor>& posv, const std::list<Functor>& negv,
			const std::map<const Functor*,const clause*>& fun2cl, Constraints& constr);
		// Detect Symmetry
		void detect_symmetry(const std::list<Functor>& posv, const std::list<Functor>& negv,
			const std::map<const Functor*,const clause*>& fun2cl, Constraints& constr);
		// Update Dynamic Constraints during covering
		void update_dynamic_constraints(const Functor& ex, const Constraints& constr);

		// Detect properties after induction
		void post_detect_properties(const std::list<Functor>& posv, const std::list<Functor>& negv);

		// Preprocessing: strip examples into PI and NI, copy signatures into SI, and return *this with ex kept
		template <typename NI>
		void generalize_preprocess(const sign_t&, std::list<clause>& pex, NI neg_out, Constraints&, std::map<const Functor*,const clause*>&);
		// Postprocessing: put back unused examples, compress, remove mode decl
		template <typename PI, typename NI>
		void generalize_postprocess(const sign_t&, std::list<clause>&, PI pbeg, PI pend, NI nbeg, NI nend);
		// General higher order function for covering algorithm
		template <class InitFun, class SearchFun, class Finalize>
		void covering(Program&, std::list<clause>&, std::list<clause>&, stat&, InitFun, SearchFun, Constraints&, Finalize, deadline_t);
		// General higher order function for generalization
		template <class InitFun, class SearchFun> 
		stat generalize(const sign_t&, InitFun, SearchFun);
		// Same as above, but allow one additional argument to finalize bsf
		template <class InitFun, class SearchFun, class Finalize> 
		stat generalize(const sign_t&, InitFun, SearchFun, Finalize);

		// Bottom Clause construction
		Functor bottom(const Functor& e, std::vector<Mode>& modemap, deadline_t deadline);
		// Determination
		bool determination(const sign_t& target, const sign_t& body_pred);

		// Compress
		// Try to remove 
		template <typename Pos, typename Neg> int compress(Pos beg, Pos end, Neg nbeg, Neg nend);
		template <typename Pi, typename Ni> int compress(const sign_t& s, Pi, Pi, Ni, Ni);

		// Read input
		void read(std::istream& is);

		// Data Members
	public:
		parameters params;
		typedef std::map<Functor,std::map<id_type,bool>> tmemo_t; // type memoization
	protected:
		code kb; // knowledge base
		modev_type modev; // modes
		// Determinations: includes and excludes
		det_type includes;
		det_type excludes;
		implications imp; // implications
		implications block; // entailments for bottom clause construction

		// Fix TRY_ME_ELSE and related pointers after adding/removing
		code::iterator fix_pointers(code::iterator c);

		// Bottom clause construction helpers
		void instantiate_one(
			const Functor& trm,
			Termindex& termindex,
			list<Functor>& botv,
			vector<Mode>& modemap,
			std::unordered_set<Functor,functor_hash_variant,functor_req>& qterms,
			set<const Functor*,functor_ptr_less>& insts,
			std::multimap<sign_t,const Functor*>& bmap,
			tmemo_t& typememo,
			deadline_t deadline);

		typedef Mode::pmarker_vec::const_iterator pmiter;
		void instantiate_input(
			Functor& build,
			const Mode& mode,
			pmiter pm_current,
			pmiter pm_ignore,
			Termindex& termindex,
			list<Functor>& botv,
			vector<Mode>& modemap,
			std::unordered_set<Functor,functor_hash_variant,functor_req>& qterms,
			set<const Functor*,functor_ptr_less>& insts,
			std::multimap<sign_t,const Functor*>& bmap,
			tmemo_t& typememo,
			deadline_t deadline);

		void instantiate_output(
			Functor& build,
			const Mode& mode,
			Termindex& termindex,
			list<Functor>& botv,
			vector<Mode>& modemap,
			std::unordered_set<Functor,functor_hash_variant,functor_req>& qterms,
			set<const Functor*,functor_ptr_less>& insts,
			std::multimap<sign_t,const Functor*>& bmap,
			tmemo_t& typememo,
			deadline_t deadline);

		// Put Instructions
		void put_structure(int f, int n, int a);
		void put_list(int xreg);
		void put_variable(int x, int a);
		void put_stack_variable(int yi, int a);
		void put_value(int x, int a);
		void put_stack_value(int yi, int a);
		void put_unsafe_value(int yi, int w);
		void put_constant(int c, int w);
		void put_int(long long, int w);
		void put_float(double, int w);
		// Set Instructions
		void set_variable(int v);
		void set_stack_variable(int yi);
		void set_value(int v);
		void set_stack_value(int yi);
		void set_local_value(int v);
		void set_local_stack_value(int yi);
		void set_void(int n);
		void set_constant(int c);
		void set_int(long long);
		void set_float(double);
		// Get Instructions
		void get_structure(int f, int n, int a);
		void get_list(int xreg);
		void get_variable(int x, int a);
		void get_stack_variable(int yi, int a);
		void get_value(int x, int a);
		void get_stack_value(int yi, int w);
		void get_constant(int c, int t);
		void get_int(long long, int t);
		void get_float(double, int t);
		// Unify Instructions
		void unify_variable(int v);
		void unify_stack_variable(int yi);
		void unify_value(int v);
		void unify_stack_value(int yi);
		void unify_local_value(int v);
		void unify_local_stack_value(int yi);
		void unify_void(int n);
		void unify_constant(int c);
		void unify_int(long long);
		void unify_float(double);
		// Register Variables for Printing
		void reg_hvar(int id);
		void reg_svar(int id, int yi);
		void reg_arg(int x);
		// Memory and Jump Instructions
		void call(int f, int n, int vars);
		void dynamic_call(int p, int vars);
		void dynamic_stack_call(int yi, int vars);
		void execute(int f, int n);
		void dynamic_execute(int p);
		void dynamic_stack_execute(int yi);
		void proceed();
		void allocate();
		void deallocate();
		void globalize_svars();
		void try_me_else(int fun);
		void retry_me_else(int k);
		void trust_me();
		// Build bindings
		bool prompt(const qconstraints& qc, answer_type& subs, int& sol_count);
		// Build solutions
		//bool prompt(int recall, const sign_t& s, std::vector<Functor>& build);
		void noop();
		void fail();
		void var(int v);
		void nonvar(int v);
		void atomic(int v);
		void atom(int v);
		void number(int v);
		void integer(int v);
		void isfloat(int v);
		void unify_instr(int v, int w);
		void nonunifiable(int v, int w);
		void equal(int v, int w);
		void unequal(int v, int w);
		inline void halt() { throw destruct_and_exit(); }
		void name(int v, int w);
		void asserta(int v);
		void assertz(int v);
		void nb_linkarg(int u, int v, int w);
		void duplicate_term(int v, int w);
		void retract_all(int v);
		void define_op(int x, int y, int z);
		void listing(bool uncompile);
		void listing(int v, bool dcomp);
		void write(int v);
		void nl();
		void consult(int v);
		void consult_list(int v, int w);
		void is(int v, int w);
		void numequal(int v, int w);
		void numunequal(int v, int w);
		void numless(int v, int w);
		void numlesseq(int v, int w);
		void numgreater(int v, int w);
		void numgreatereq(int v, int w);
		void univ(int v, int w);
		void functor(int x, int y, int z);
		void arg(int x, int y, int z);

		void compare(int x, int y, int z);
		void before(int v, int w);
		void before_eq(int v, int w);
		void after(int v, int w);
		void after_eq(int v, int w);
		void variant(int v, int w);
		void notvariant(int v, int w);

		void neck_cut();
		void get_level(int yi);
		void cut(int yi);

		void modeh(int v, int w);
		void modeb(int v, int w, int s);
		void add_constraint(int v, int w, bool is_bottom);
		void set_param(int v, int w);
		void get_param(int v, int w);
		void determination(int v, int w, bool is_include);
		void generalize();
		void generalize(int v);

		void reset(code& kb, std::vector<Mode>& mv);
		void backtrack();
		//void reset();
	};
	// Print Program
	std::ostream& operator<<(std::ostream&, const Program&);
	inline std::istream& operator>>(std::istream& is, Program& p) { p.read(is); return is; }

	namespace {
		lp::concept con_dummy;
	}


	inline auto Program::concept_range(const lp::sign_t& p) 
		-> std::pair<cl_iterator,cl_iterator>
	{
		auto at = kb.find(p);
		if (at == kb.end()) return std::make_pair(con_dummy.end(),con_dummy.end());
		return std::make_pair(at->second.begin(),at->second.end());
	}

	inline auto Program::concept_range(const lp::sign_t& p) const 
		-> std::pair<cl_const_iterator,cl_const_iterator>
	{
		auto at = kb.find(p);
		if (at == kb.end()) return std::make_pair(con_dummy.end(),con_dummy.end());
		return std::make_pair(at->second.begin(),at->second.end());		
	}

	inline void Program::push_front(const Functor& t)
	{
		if (verbosity.is_int() && verbosity.get_int() >= debug_type::warning && builtin.find(t.signature()) != builtin.end()) {
			DEBUG_WARNING(std::cerr << "Warning: asserta-ing a predicate that looks like a builtin " << t << " (did you mean to use a query?)\n");
		}
		this->compile_insert(t,false,false);
	}

	inline void Program::push_back(const Functor& t)
	{
		if (verbosity.is_int() && verbosity.get_int() >= debug_type::warning && builtin.find(t.signature()) != builtin.end()) {
			DEBUG_WARNING(std::cerr << "Warning: asserting a predicate that looks like a builtin " << t << " (did you mean to use a query?)\n");
		}
		this->compile_insert(t,false,true);
	}


	inline auto Program::erase_concept(concept_iterator ci) -> concept_iterator
	{
		return kb.erase(ci);
	}

	inline void Program::erase_clause(concept_iterator cc, cl_iterator cl)
	{
		cc->second.erase(cl);
		fix_pointers(cc);
	}

	inline void Program::erase_clause(const lp::sign_t& p, cl_iterator cl)
	{
		assert( kb.find(p) != kb.end() );
		return erase_clause(kb.find(p),cl);
	}


	template <typename PI, typename NI>
	void Program::examples_to_functors(const sign_t& sign, PI pi, NI ni, bool strip_negatives)
	{
		// What constitutes an example is based on the signature of the modeh's
		for (auto mh = modev.begin(); mh != modev.end(); ++mh) {
			//std::cerr << "Mode: " << *mh << "\n";
			if (mh->is_head() && mh->atom().signature() == sign) { // modeh
				if (!mh->atom().is_function() && !mh->atom().is_constant()) {
					std::cerr << "Error: modeh declaration is non-function\n";
					assert(false);
					continue;
				}
				// Check that target predicate isn't builtin or already defined
				if (lp::builtin.find(sign) != lp::builtin.end()) {
					std::cerr << "Error: trying to induce builtin: " << fmap.get_data(sign.first) << "/" << sign.second << "\n";
					throw builtin_overwrite();
				}
				DEBUG_INFO(cout << "Adding target predicate: " << fmap.get_data(sign.first) << "/" << sign.second << "\n");
				// Copy signature and all affected examples
				// Positive examples
				// Note we convert positive examples into queries
				auto at = kb.find(sign);
				if (at != kb.end()) {
					auto is_pos_ex = [](const sign_t& s, const clause& cl){ return body_literals(cl) == 0 && is_ground(s,cl); };
					std::for_each(at->second.begin(),at->second.end(),[&](clause& cl){
						if (is_pos_ex(sign,cl)) {
							*pi++ = decompile(sign,cl);
						} else {
							DEBUG_INFO(std::cerr << "Ignoring non-positive example: " << decompile(sign,cl) << ")\n");
						}
					});
					auto ri = std::remove_if(at->second.begin(),at->second.end(),[&](const clause& cl){ return is_pos_ex(sign,cl); });
					at->second.erase(ri,at->second.end());
					fix_pointers(at);
				}

				// Negative examples / Constraints
				at = kb.find(std::make_pair(if_id,1));
				if (at != kb.end()) {
					for (auto e = at->second.begin(); e != at->second.end(); ++e) {
						*ni++ = decompile(*e);
					}
					// Note: we do not erase negative examples as they may be needed in future inductions (e.g. when inducing multiple predicates)
					if (strip_negatives) {
						auto ri2 = std::remove_if(at->second.begin(),at->second.end(),[&](const clause& cl){ return decompile(cl).arg_first()->signature() == sign; });
						at = kb.find(std::make_pair(if_id,1));
						at->second.erase(ri2,at->second.end());
						fix_pointers(at);
					}
				}
			} // for each modeh
		} // for each mode declaration
	}


	template <typename PI, typename NI>
	void Program::examples_to_queries(const sign_t& sign, PI pi, NI ni, bool strip_negatives)
	{
		//std::vector<sign_t> sv;
		std::deque<Functor> pv,nv;
		examples_to_functors(sign,std::back_inserter(pv),std::back_inserter(nv), strip_negatives);
		//for (auto&& s : sv) *si++ = std::move(s);
		for (auto& f : pv) {
			Functor q(if_id);
			q.attach_child(&f);
			try {
				*pi++ = compile(q);
			} catch (...) {
				q.unattach_child();
				throw;
			}
			q.unattach_child();
		}
		for (auto& f : nv) *ni++ = compile(f);
	}

	template <typename MI>
	inline void group_examples(MI beg, MI end, std::list<Functor>& posv)
	{
		std::map<sign_t,int> sign2ord;
		//std::map<int,sign_t> ord2sign;
		int k = 0;
		for (auto m = beg; m != end; ++m) {
			if (!m->is_head()) continue;
			const sign_t s = m->atom().signature();
			auto at = sign2ord.find(s);
			if (at != sign2ord.end()) continue;
			++k;
			sign2ord.insert(std::make_pair(s,k));
			//ord2sign.insert(std::make_pair(k,s));
		}
		auto mode_before = [&](const Functor& f, const Functor& g){
			return sign2ord.find(f.signature())->second < sign2ord.find(g.signature())->second;
		};
		posv.sort(mode_before); // stable sort
	}

	template <typename PI>
	inline std::list<clause> posex_to_queries(
		PI beg, 
		PI end, 
		std::map<const Functor*,const clause*>& fun2cl)
	{
		std::list<clause> cll;
		// Compile down to clauses
		for ( ; beg != end; ++beg) {
			Functor q(if_id);
			try {
				q.attach_child(&*beg);
				cll.push_back(compile(q));
				fun2cl.insert(std::make_pair(&*beg,&cll.back()));
			} catch (...) {
				q.unattach_child();
				throw;
			}
			q.unattach_child();
		}
		return cll;
	}


	inline bool equal_except(const Functor::position& pos, const Functor* f, const Functor* g, Functor::position& cp)
	{
		if (pos == cp) {
			// Must be unequal
			if (*f == *g) return false;
		} else {
			const sign_t& fs = f->signature();
			if (fs != g->signature()) return false;
			for (int k = 0; k < fs.second; ++k) {
				cp += k;
				if (!equal_except(pos,f->arg(k),g->arg(k),cp)) return false;
				cp.pop_back();
			}
		}
		return true;
	}
	inline bool equal_except(const Functor::position& pos, const Functor* f, const Functor* g) { Functor::position cp; return equal_except(pos,f,g,cp); }


	template <typename NI>
	void Program::generalize_preprocess(
		const sign_t& sign,
		std::list<clause>& pex, 
		NI neg_beg, 
		Constraints& constr,
		std::map<const Functor*,const clause*>& fun2cl)
	{
		//===== Preprocessing: Get signatures and Example sets =====//

		// Make a theory of current DB by copying it and removing all relevant examples
		// (*this) will be used for bottom clause construction (so the examples must be present)
		std::list<Functor> posv,negv;
		this->examples_to_functors(sign,std::back_inserter(posv),std::back_inserter(negv));

		// Detect various patterns
		detect_dependent(posv,negv,fun2cl,constr);
		detect_reflexivity(posv,negv,fun2cl,constr);
		detect_symmetry(posv,negv,fun2cl,constr);

		// Reorder examples?
		const data& dat = params[parameters::pos_order];
		if (dat.is_atom() && strcmp(dat.get_atom(),"sequential") == 0) {
			// Do nothing
		} else if (dat.is_atom() && strcmp(dat.get_atom(),"random") == 0) {
			// random shuffle
			DEBUG_INFO(cout << "Randomly shuffling positive examples\n");
			std::vector<Functor> vec; vec.reserve(posv.size());
			for (auto&& f : posv) vec.push_back(std::move(f));
			posv.clear();
			std::random_shuffle(vec.begin(),vec.end());
			for (auto&& f : vec) posv.push_back(std::move(f));
		} else if (dat.is_atom() && strcmp(dat.get_atom(),"complexity") == 0) {
			// increasing by complexity
			DEBUG_INFO(cout << "Ordering positive examples by complexity\n");
			posv.sort([](const Functor& f, const Functor& g){ return f.size() < g.size(); });
		} else if (dat.is_atom() && strcmp(dat.get_atom(),"regularity") == 0) {
			auto score = [&](const Functor& cl) -> double {
				auto rng = constr.uncond.equal_range(cl);
				double s = 0;
				for (auto i = rng.first; i != rng.second; ++i) {
					s += 1.0 / (1.0 + i->first.size()); // shorter antecedent => better
				}
				auto rng2 = constr.cond.equal_range(cl);
				for (auto i = rng2.first; i != rng2.second; ++i) {
					s += 1.0 / (1.0 + i->first.size()); // shorter antecedent => better
				}
				return s;
			};
			auto cmp = [&](const Functor& f, const Functor& g){ return score(f) > score(g); };
			posv.sort(cmp);
			//std::sort(posv.begin(),posv.end(),cmp);
			//for (const auto& f : posv) {
			//	std::cerr << "  " << f << " (" << score(f) << ")\n";
			//}
		}

		// Regroup examples according to order of mode declarations
		group_examples(modev.begin(),modev.end(),posv);

		// Compile positive examples into queries
		pex = posex_to_queries(posv.begin(),posv.end(),fun2cl);
		// Compile negative examples into queries
		for (auto& f : negv) {
			*neg_beg = compile(f);
			//fun2cl.insert(std::make_pair(&*neg_beg));
			++neg_beg;
		}


		// Clear graph file?
		if (!params.is_unset(parameters::graph)) {
			ofstream gout;
			string fname;
			if (params[parameters::graph].is_atom()) {
				fname = params[parameters::graph].get_atom();
				if (fname.substr(fname.length()-4) != ".dot") fname += ".dot";
			} else {
				fname = "graph.dot";
			}
			gout.open(fname.c_str(),ios::trunc);
			if (!gout.is_open()) {
				DEBUG_CRUCIAL(std::cerr << "Warning: could not open graph file " << fname << "\n");
			}
			gout << "digraph G {\n";
		}
	}


	template <typename PI, typename NI>
	void Program::generalize_postprocess(const sign_t& sign, std::list<clause>& pex, PI xbeg, PI xend, NI nbeg, NI nend)
	{
		DEBUG_TRACE(std::cerr << "Putting back positive examples\n");
		// Put back all positive examples that were not generalized
		for (auto& cl : pex) {
			// Note: examples are in negated form
			Functor e = decompile(cl);
			assert( e.is_function(if_id,1) );
			this->push_back(*e.arg_first());
		}

		// Compress according to compression level
		const auto clevel = params.force_int(parameters::compress);
		if (clevel > 0) {
			DEBUG_TRACE(std::cerr << "Compressing...\n");
			if (clevel >= 2) {
				// Filtered compression: remove all positives that are not covered first
				std::vector<clause> fex;
				//for (auto i = xbeg; i != xend; ++i) {
				//	if (this->covers(*i)) {
				//		//std::cerr << "Filter: covered " << decompile(*i) << "\n";
				//		fex.push_back(*i);
				//	} else {
				//		//std::cerr << "Filter: NOT covered " << decompile(*i) << "\n";
				//	}
				//}
				std::copy_if(xbeg,xbeg,std::back_inserter(fex),[&](const clause& cl){ return this->covers(cl); });
				if (!fex.empty()) {
					this->compress(sign,fex.begin(),fex.end(),nbeg,nend);
				} else {
					// cannot use filtered compression
					DEBUG_WARNING(std::cerr << "Warning: cannot use filtered compression since no example is covered anymore\n");
				}
			} else {
				this->compress(sign,xbeg,xend,nbeg,nend);
			}
		}

		// Flatten
		if (params.is_set(parameters::flatten)) {
			DEBUG_TRACE(std::cerr << "Flattening...\n");
			//std::cerr << "Flatten, s: " << fmap.get_data(s.first) << "/" << s.second << "\n";
			auto at = kb.find(sign);
			if (at != kb.end()) { 
				//std::cerr << "  found? " << (at != kb.end()) << "\n";
				auto& con = at->second;
				for (auto& cl : con) {
					Functor t = decompile(sign,cl);
					//std::cerr << "Original:  " << t << "\n";
					// Also Flatten clauses
					t.flatten();
					//std::cerr << "Flattened: " << t << "\n";
					//std::cerr << "Print all:\n"; print_all(std::cerr,t.get()); std::cerr << "\n";
					// Make collectable
					cl.clear();
					cl.push_back(instruction(instruction::NOOP)); // for RETRY instr in fix_pointers
					make_wam_instructions(t,false,&cl);
				}
				fix_pointers(at);
			}
		}

		// Detect symmetry/reflexivity/idempotence and alter modeb
		if (params.is_set(parameters::post_detect)) {
			std::list<Functor> posv,negv;
			std::for_each(xbeg,xend,[&](const clause& cl){ posv.push_back(*decompile(cl).arg_first()); });
			std::for_each(nbeg,nend,[&](const clause& cl){ negv.push_back(*decompile(cl).arg_first()); });
			post_detect_properties(posv,negv);
		}

		// Finish graph file?
		if (!params.is_unset(parameters::graph)) {
			ofstream gout;
			string fname;
			if (params[parameters::graph].is_atom()) {
				fname = params[parameters::graph].get_atom();
				if (fname.substr(fname.length()-4) != ".dot") fname += ".dot";
			} else {
				fname = "graph.dot";
			}
			gout.open(fname.c_str(),ios::app);
			if (!gout.is_open()) {
				DEBUG_CRUCIAL(std::cerr << "Warning: could not open graph file " << fname << "\n");
			}
			gout << "}\n";
		}

		// Clear up the modeh's
		auto r_at = std::remove_if(modev.begin(),modev.end(),[&](const Mode& m){ return m.is_head() && m.atom().signature() == sign; });
		modev.erase(r_at,modev.end());

		DEBUG_TRACE(std::cerr << "Post-Processing done\n");
	}


	//======================= Induction Templates =====================//

	template <class InitFun, class SearchFun>
	stat Program::generalize(const sign_t& sign, InitFun init, SearchFun search)
	{
		typedef decltype(init(*this,Functor(),1,vector<Mode>(),list<clause>(),list<clause>(),Constraints(),lp::stat())) candidate_type;
		auto finalize = [](const Program&, list<clause>&, list<Program::clause>&, Functor&, 
			int, const vector<Mode>&, bsf_type&, lattice<candidate_type>&, stat&) -> void {};
		return this->generalize(sign,init,search,finalize);
	}

	template <class InitFun, class SearchFun, class Finalize>
	stat Program::generalize(const sign_t& sign, InitFun init, SearchFun search, Finalize finalize)
	{
		// Measure time for induction
		clock_t uptime = clock();
		const double time_limit = params.force_float(parameters::time_limit);
		const auto deadline = (time_limit == std::numeric_limits<double>::max() ? deadline_t::max() : std::chrono::steady_clock::now() + std::chrono::milliseconds(long long(1000.0*time_limit)));
		// Statistics
		stat stats;

		// Keep a theory with all examples
		Program theory_with_examples = *this;

		//===== Preprocessing: Get signatures and extract Example sets =====//
		std::list<clause> pex;
		std::list<clause> nex;
		//std::cerr << "Cap: " << *cap << "\n";
		//std::cerr << "Generalize preprocess...\n";
		Constraints constr;
		std::map<const Functor*,const clause*> fun2cl;
		generalize_preprocess(sign,pex,back_inserter(nex),constr,fun2cl);
		// Generalize_preprocess might create new constraints
		theory_with_examples.block = this->block;
		theory_with_examples.imp = this->imp;

		//std::cerr << "Preprocess done\n";
		// Remember original number of positive and negative examples
		const int orig_pex_size = pex.size();
		const int orig_nex_size = nex.size();
		// Backup all examples into 'ex' (used to compress theory)
		std::vector<clause> pex2;
		pex2.reserve(pex.size());
		std::copy(pex.begin(),pex.end(),std::back_inserter(pex2));

		//std::cerr << "Displaying all examples:\n"; for (auto& ex : pex) std::cerr << decompile(ex) << "\n";

		DEBUG_INFO( cout << "We have " << orig_pex_size << " positive and " << orig_nex_size << " negative examples\n" );
		//std::cerr << "Cap: " << *cap << "\n";

		//===== Set Covering Algorithm =====//
		try {
			this->covering(theory_with_examples,pex,nex,stats,init,search,constr,finalize,deadline);
		} catch (induction_aborted) {
			std::cerr << "WARNING: aborting induction, post-processing now\n";
			++stats[stat::full_abort];
		} catch (time_out) {
			std::cerr << "WARNING: aborting induction due to time out, post-processing now\n";
			++stats[stat::full_abort];
		} catch (std::bad_alloc) {
			std::cerr << "ERROR: ran out of memory during induction, post-processing now\n";
		}
		// Note: pex has been reduced to examples that were not covered

		//===== Postprocessing: search done, cleanup and compress DB =====//
		//std::cerr << "Generalize postprocess...\n";
		//std::cerr << "Positives: " << pex.size() << "\n";
		generalize_postprocess(sign,pex,
			pex2.begin(),pex2.end(),nex.begin(),nex.end());
		//std::cerr << "Generalize postprocess Done\n";
		pex.clear();

		//===== Compute some statistics =====//

		// Execution time (in ms) for this algorithm
		stats[stat::time] = useful::round<int>(1000.0 * ((clock()-uptime) / double(CLOCKS_PER_SEC)));

		// Compute complexity
		stats[stat::complexity] = 0;
		const auto r = this->concept_range(sign);
		for (auto cl = r.first; cl != r.second; ++cl) {
			//std::cerr << "Found one ex_sign...\n";
			stats[stat::complexity] += 1 + lp::body_literals(*cl); // + 1 for head
		}

		// Print solution and statistics
		DEBUG_CRUCIAL(
			cout << "\nSolution:\n";
			for_each(r.first,r.second,[&](const clause& cl){ 
				//if (params.is_set(parameters::print_labels)) cout << cl.label() << " | ";
				//std::cerr << "Decompiling...\n";
				cout << "  " << decompile(sign,cl) << ".\n"; 
			});
		cout << "Positive examples:           " << orig_pex_size << "\n";
		cout << "Negative examples:           " << orig_nex_size << "\n";
		cout << "Clauses Added:               " << stats[stat::cl_added] << "\n";
		cout << "Partial Fitness Evaluations: " << stats[stat::fitc] << "\n";
		cout << "Nodes Explored:              " << stats[stat::nodec] << "\n";
		cout << "Examples Evaluated:          " << stats[stat::exc] << "\n";
		cout << "Consistent/Inconsistent:     " << stats[stat::conc] << "/" << stats[stat::inconc] << "\n";
		cout << "Percentage consistent:       " << 100.0*stats[stat::conc]/(stats[stat::conc]+stats[stat::inconc]) << "%\n";
		cout << "Complexity:                  " << stats[stat::complexity] << "\n";
		cout << "Bottom clauses:              " << stats[stat::bottomc] << "\n";
		cout << "Searches exhausted           " << stats[stat::exhausted] << "\n";
		cout << "Induction time (s):          " << stats[stat::time]/1000.0 << "\n";
		);

		return stats;
	}


	template <class InitFun, class SearchFun, class Finalize>
	void Program::covering(
		Program& theory_with_examples,
		std::list<clause>& pex,
		std::list<clause>& nex,
		stat& stats,
		InitFun init_search,
		SearchFun find_candidate,
		Constraints& constr,
		Finalize finalize,
		deadline_t deadline)
	{
		typedef decltype(init_search(*this,Functor(),1,vector<Mode>(),pex,nex,Constraints(),stats)) candidate_type;
		// Statistics
		lattice<candidate_type> grid;
		std::map<const clause*,double> neg_count;
		// Keep track of positives used for generalization
		//std::deque<std::pair<Functor,int>> pos_score;

		// Cache parameters
		const int param_noise = params.force_int(parameters::noise);
		const bool param_graph = !params.is_unset(parameters::graph);
		const int param_nodes = params.force_int(parameters::nodes);
		const int param_max_nodes = params.force_int(parameters::max_nodes);
		const int param_inflate = params.force_int(parameters::inflate);
		const int param_max_bottom = params.force_int(parameters::max_bottom);
		const int param_reorder_negs = params.force_int(parameters::reorder_negs);
		const int param_compress = params.is_set(parameters::compress);
		const double param_solution_if = params.force_float(parameters::solution_if);

		// Perform non-sequential search?
		//const bool param_detect_dependent = params.is_set(parameters::detect_dependent);
		//const bool param_detect_reflexivity = params.is_set(parameters::detect_reflexivity);
		//const bool param_detect_symmetry = params.is_set(parameters::detect_symmetry);
		//const bool seq_covering = (!param_detect_dependent && !param_detect_reflexivity && !param_detect_symmetry);

		assert( stats[stat::exc] == 0 );
		// Backup pex
		const auto pex2 = pex;
		// Number of original positive examples
		const int orig_pex_size = pex.size();

		auto sum_nodes = [](const stat& st) -> double { return st[stat::invalid] + st[stat::inconc] + st[stat::conc]; };
		bool fail = false;
		auto termination_cond = [&]() -> bool { 
			return fail || sum_nodes(stats) > param_max_nodes || stats[stat::bottomc] > param_max_bottom || std::chrono::steady_clock::now() > deadline;
		};

		while (!pex.empty() && !termination_cond()) {
			const clause* cl_ex = &pex.front();
			Functor nege = decompile(*cl_ex);
			assert(nege.is_function(if_id,1));
			const Functor& ex = *nege.arg_first();
			// Compute bottom clause (from front example) and using original DB (we need all examples)
			std::vector<Mode> modes; // modes will store all literals and their modes
			//std::cerr << "Before bottom(), Cap: " << *cap << "\n";
			DEBUG_INFO( cout << "Computing bottom clause from example: " << ex << "\n" );
			++stats[stat::bottomc];

			// Update dynamic constraints
			update_dynamic_constraints(ex,constr);
			// Update theory_with_examples too
			theory_with_examples.block = this->block;
			theory_with_examples.imp = this->imp;
			
			reset_mask_bottom();
			Functor bot = theory_with_examples.bottom(ex,modes,deadline);
			//std::cerr << "Bottom clause constructed\n";
			DEBUG_INFO(std::cerr << "Bottom Clause Literals: " << bot.body_size() << "\n");
			DEBUG_INFO(std::cerr << "Bottom Clause Size: " << bot.size() << "\n");
			DEBUG_INFO(std::cerr << "Bottom Clause: " << bot << "\n");
			DEBUG_TRACE( string s6; cin.sync(); getline(cin,s6); );
			if (bot.is_nil()) {
				DEBUG_WARNING(cout << "warning: could not create bottom clause from example\n");
				DEBUG_TRACE( string s6; cin.sync(); getline(cin,s6); );
				//++xi;
				//continue;
			}
			//std::cerr << "****** Bottom clause: " << bot << "\n";
			// Size of bottom clause
			const int BOT_BSIZE = int(bot.body_size());

			DEBUG_INFO(
				cout << "Bottom Clause: " << bot << "\n";
			cout << "Body size:     " << BOT_BSIZE << "\n";
			cout << "From example:  " << ex << "\n";
			);
			//DEBUG_TRACE( string s6; cin.sync(); getline(cin,s6); );

			// Initialize iteration
			candidate_type bsf;
			if (!bot.is_nil()) {
				try { // init_search may throw bottom_rejected
					//std::cerr << "Before init_search, Cap: " << *cap << "\n";
					bsf = init_search(*this,bot,BOT_BSIZE,modes,pex,nex,constr,stats);
					// Initialize graph
					if (param_graph) {
						// cerr << "Initializing graph...\n";
						grid.clear();
						grid.bottom(bot,ex);
						grid.name(std::to_string(stats[stat::bottomc]));
						// cerr << "graph init done\n";
					}

					// Start Search
					try { // may throw search_completed / search_aborted
						bool finished = false;
						for (int nodec = 0; nodec < param_nodes && sum_nodes(stats) < param_max_nodes && !finished; ) {
							if (std::chrono::steady_clock::now() > deadline) throw time_out();
							candidate_type sol;
							try {
								sol = find_candidate(*this,pex,nex,bot,BOT_BSIZE,modes,bsf,constr,stats,deadline);
							} catch (const candidate_type& last_cand) { // catch by rvalue not allowed
								DEBUG_INFO(cout << "Premature termination from fitness evaluation\n");
								sol = last_cand;
								++stats[stat::exhausted];
								finished = true;
							}

							// Note: find_candidate has already called mask_bottom

							DEBUG_INFO(
								std::cout << "(" << sol.fitness() << "," << sol.pos() << "," << sol.neg() << ") ";
								print_candidate(std::cout,bot); std::cout << "\n";
								);

							// Print Solution into graph?
							if (param_graph) {
								// cerr << "graph inserting: " << sol.candidate() << "\n";
								grid.insert(sol,bot);
								// cerr << "graph insertion done\n";
							}

							// Update node count if sol is valid = fitness is not -infty
							if (sol.is_valid()) {
								// Update statistics
								++stats[stat::nodec];
								++nodec;
								if (param_reorder_negs > 0) {
									// Update counter for negatives covered (note: this may apply to consistent solutions with noise)
									const auto& ncov = sol.ncover();
									if (!ncov.empty()) {
										const double scale = params.force_float(parameters::reorder_negs_factor);
										for (auto& p : ncov) {
											const double value = 1.0/(1.0 +  scale * p.second);
											//std::cerr << "Value: " << value << " = 1.0/(1.0 + " << scale << " * " << p.second << ")\n";
											auto at = neg_count.find(&*p.first);
											if (at == neg_count.end()) {
												// New
												neg_count.insert(std::make_pair(&*p.first,value));
											} else {
												// Example already available
												at->second += value;
											}
										}
										// Reorder negative examples
										nex.sort([&](const clause& x, const clause& y){ return neg_count[&x] > neg_count[&y]; });
									}
								}

								// Update best so far
								if (sol.is_consistent(param_noise)
									&& (bsf.is_nil() || bsf.fcompare(sol,param_noise) < 0)
									&& 100.0 * std::count(sol.mask().begin(),sol.mask().end(),1) / sol.pos() < param_inflate) {
										DEBUG_TRACE( cerr << "Solution replaces best so far: " << bsf.mask() << " with fitness " << bsf.fitness() << "\n" );
										bsf = std::move(sol);
										// Check termination criterion
										if (double(bsf.pos()) / double(orig_pex_size) > double(param_solution_if)) {
											throw search_aborted();
										}
								}
							}
						} // while there are nodes left to explore
					} catch (search_completed) {
						// No more solutions
						DEBUG_INFO( std::cerr << "Search terminated\n" );
						//DEBUG_TRACE( string s6; cin.sync(); getline(cin,s6); );
						++stats[stat::exhausted];
					} catch (search_aborted) {
						// Reached max examples evaluated limit (or DPLL limit)
						DEBUG_INFO( std::cerr << "Search aborted\n" );
						++stats[stat::aborted];
						//DEBUG_TRACE( string s6; cin.sync(); getline(cin,s6); );
						//unmask_bottom(bot);
						//break; // abort search
					}

					// Convert bottom to bsf candidate
					if (!bsf.is_nil()) {
						mask_bottom(bot,bsf.mask());
						// Finalize best so far if there is such a functions (not used by most algorithms)
						finalize(*this,pex,nex,bot,BOT_BSIZE,modes,bsf,grid,stats);
					}

					// Print Graph
					if (param_graph) {
						ofstream gout;
						string fname;
						if (params[parameters::graph].is_atom()) {
							fname = params[parameters::graph].get_atom();
							if (fname.substr(fname.length()-4) != ".dot") fname += ".dot";
						} else {
							fname = "graph.dot";
						}
						gout.open(fname.c_str(),ios::app);
						if (!gout.is_open()) {
							DEBUG_CRUCIAL(std::cerr << "Warning: could not open graph file " << fname << "\n");
						}
						grid.compute_rank(param_noise, param_inflate);
						grid.print_subgraph(gout,params,&modes);
						gout.close();
					}

				} catch (bottom_rejected) {
					DEBUG_BASIC( cout << "Ignoring Bottom Clause: " << bot << "\n" );
					//DEBUG_TRACE( string s6; cin.sync(); getline(cin,s6); );
				}

			} // if bot is valid (not nil)

			// Finished search for this bottom clause

			// Perform action (add candidate, add example)
			//std::cerr << "Explored candidates: " << stats[stat::nodec] << "\n";
			if (bot.is_nil() || bsf.is_nil()) {
				// Just add example
				if (bot.is_nil()) {
					DEBUG_BASIC(cout << "Could not construct bottom clause, adding example instead: " << ex << "\n");
				} else if (bsf.is_nil()) {
					DEBUG_BASIC(cout << "Could not find viable candidate, adding example instead: " << ex << "\n");
				}
				//theory_with_examples.push_back(fe); // insert into this for better bottom clause construction
				this->push_back(ex); // add to theory
				++stats[stat::cl_added]; // update number of changed clauses
				pex.pop_front();
			} else {
				// We have a valid best-so-far
				assert(bsf.is_consistent(param_noise));
				DEBUG_BASIC(std::cout << "Adding Candidate: (" << bsf.fitness() << "," << bsf.pos() << "," 
					<< bsf.neg() << "): "; print_candidate(std::cout,bot); std::cout << "\n");

				// Update number of changed clauses
				++stats[stat::cl_added];

				// Insert solution into *this for better bottom clause construction
				theory_with_examples.push_back(bot);

				// Insert solution into theory
				//std::cerr << "Adding candidate: " << bot << "\n";
				this->push_back(bot);

				//pos_score.push_back(std::make_pair(*decompile(ex).arg_first(), bsf.pos()));

				// Remove covered positive examples (stored in pcover)
				for (auto i : bsf.pcover()) {
					pex.erase(i);
				}
				DEBUG_INFO(cout << "Removed examples: " << bsf.pcover().size() << "\n");
				//cout << "Remaining examples: " << pex.size() << "\n";

				// Compress
				if (param_compress) {
					const int compressed = this->compress(bot.head()->signature(),pex2.begin(),pex2.end(),nex.begin(),nex.end());
					// Check coverage once more
					if (compressed > 0) {
						int rem = 0;
						for (auto i = pex.begin(); i != pex.end(); ) {
							if (this->covers(*i,nullptr,nullptr,&deadline)) {
								++rem;
								i = pex.erase(i);
							} else ++i;
						}
						DEBUG_INFO(cout << "Removed additional examples: " << rem << "\n");
					}
				}
			} // if candidate was added

			// Restore bottom clause (to ensure proper deletion)
			unmask_bottom(bot);

//#ifndef NDEBUG
//			// Pause
//			std::cerr << "Examples evaluated: " << stats[stat::exc] << "\n";
//			std::cerr << "Nodes explored: " << stats[stat::nodec] << "\n";
//			std::cerr << "Listing: " << *this << "\n";
//			string intmp;
//			cin.sync();
//			cerr << "Press enter to continue\n";
//			//getline(cin,intmp);
//#endif
		} // for each positive example

		// Print statistics about positive examples
		//for (const auto& p : pos_score) {
		//	std::cerr << "Important Example: " << p.first << " -> " << p.second << "\n";
		//}

		// Print statistics about negative examples
		//if (param_reorder_negs) {
		//	typedef decltype(neg_count) neg_count_type;
		//	typedef const neg_count_type::value_type* pptr_t;
		//	std::vector<pptr_t> nord;
		//	nord.reserve(neg_count.size());
		//	double tot_count = 0;
		//	for (const auto& p : neg_count) {
		//		tot_count += p.second;
		//		nord.push_back(&p);
		//	}
		//	std::sort(nord.begin(),nord.end(),[](pptr_t x, pptr_t y){ return x->second > y->second; });
		//	cout << std::left;
		//	std::cout << "Negative Examples evaluated: " << neg_count.size() << " (count: " << int(tot_count) << ")\n";
		//	std::cout << right << setw(5) << "PDF" << setw(5) << "CDF" << setw(15) << "Count" 
		//		<< setw(5) << " " << left << setw(30) << "Negative Example" << "\n";
		//	double cdf = 0;
		//	auto to_perc = [](double d) -> int { return int(d*100.0 + 0.5); };
		//	for (auto ptr : nord) {
		//		stringstream ss;
		//		ss << decompile(*ptr->first);
		//		const double pr = ptr->second / tot_count;
		//		cdf += pr; 
		//		std::cout << right << setw(5) << to_perc(pr) << setw(5) << to_perc(cdf) << setw(15) << ptr->second 
		//			<< setw(5) << " " << left << setw(30) << ss.str() << "\n";
		//	}
		//}
		// End of negative examples statistics

		if (sum_nodes(stats) > param_max_nodes) {
			DEBUG_WARNING(std::cerr << "Warning: global maximum of " << param_max_nodes << " nodes explored\n");
			throw induction_aborted();
		}
		if (stats[stat::bottomc] > param_max_bottom) {
			DEBUG_WARNING(std::cerr << "Warning: maximum of " << param_max_bottom << " bottom clauses constructed\n");
			throw induction_aborted();
		}

	}


	template <class Iter>
	int Program::covers_clauses(const Functor& cand, Iter beg, Iter end, deadline_t* deadline)
	{
		// Insert candidate
		int res = 0;
		const sign_t sc = cand.head()->signature();
		push_back(cand);
		try {
			for ( ; beg != end; ++beg) {
				//std::set<id_type> v;
				res += query(*beg,deadline);
			}
		} catch (...) {
			// Erase candidate before rethrowing
			this->pop_back(sc);
			throw;
		}
		// Erase candidate
		this->pop_back(sc);
		return res;
	}

	template <typename PIter, typename NIter>
	int Program::compress(PIter pbeg, PIter pend, NIter nbeg, NIter nend)
	{
		int count = 0;
		for (auto i = kb.begin(); i != kb.end(); i = kb.upper_bound(i->first)) {
			count += this->compress(i->first,pbeg,pend,nbeg,nend);
		}
		return count;
	}

	template <typename PIter, typename NIter>
	int Program::compress(
		const sign_t& sg,
		PIter beg, PIter end,
		NIter nbeg, NIter nend)
	{
		// Try to remove one s/a and check coverage
		int count = 0;
		auto at = kb.find(sg);
		bool is_empty = false;
		if (at != kb.end()) {
			concept& con = at->second;
			for (auto c = con.begin(); c != con.end(); ) {
				assert(!con.empty());
				if (con.size() == 1) break; // don't remove last clause (removing this causes problems with require_definitions)
				// Remove it first, then reinsert it if necessary
				const clause cl = std::move(*c);
				//std::cerr << "Can we remove? " << decompile(sg,&cl.front()) << "\n";
				auto cnext = con.erase(c);
				if (con.empty()) is_empty = true;
				fix_pointers(at);

				// Check positive coverage
				bool can_remove = std::all_of(beg,end,[&](const clause& c){ return this->covers(c); });
				//for (auto pi = beg; pi != end; ++pi) {
				//	//std::cerr << "Positive covered? " << decompile(*e) << "\n";
				//	if (!this->covers(*pi)) {
				//		std::cerr << "No, positive not covered: " << decompile(*pi) << "\n";
				//		// positive not covered: fail
				//		can_remove = false;
				//		break;
				//	} else {
				//		//std::cerr << "Yes, covered\n";
				//	}
				//}
				if (can_remove) {
					// Try negatives
					can_remove = std::none_of(nbeg,nend,[&](const clause& c){ return this->covers(c); });
					//for (auto ni = nbeg; ni != nend; ++ni) {
					//	if (this->covers(*ni)) {
					//		std::cerr << "No, negative covered: " << decompile(*ni) << "\n";
					//		can_remove = false;
					//		break;
					//	}
					//}
				}
				if (!can_remove) {
					// Put back
					//std::cerr << "No, we cannot remove it\n";
					if (is_empty) {
						concept cpt;
						cpt.push_back(std::move(cl));
						at = kb.insert(std::make_pair(sg,std::move(cpt))).first;
					} else {
						c = con.insert(cnext,std::move(cl));
						++c;
					}
					fix_pointers(at);
					// Note: do not set is_empty=false since there is nothing more to check
					// and iterator c is no longer valid after removal
				} else {
					// else: has already been removed
					//std::cerr << "Yes, we can remove it\n";
					++count;
					c = cnext; // move to next
				}

				if (is_empty) break;
			} // for

		} // at != kb.end()
		return count;
	}


	// Inline WAM definitions

	inline void Program::put_variable(int x, int a)
	{
		lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
		var_regs[a] = &lp::heap[heap_i];
		var_regs[x] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::put_stack_variable(int yi, int a)
	{
		// Permanent variable: Stack usage
		//std::cerr << "  put_stack_variable " << yi << ", " << a << " => " << &lp::stack[env_i + yi + 3] << "\n";
		const int i = env_i + yi + 3;
		lp::stack[i] = term::make_variable(&lp::stack[i]);
		var_regs[a] = &lp::stack[i];
		++cap;
		assert(lp::stack[env_i + yi + 3].is_variable());
	}

	inline void Program::put_value(int v, int a)
	{
		var_regs[a] = var_regs[v];
		++cap;
	}

	inline void Program::put_stack_value(int yi, int a)
	{
		//std::cerr << "  put_stack_value " << yi << ", " << a << " => " << &lp::stack[env_i + yi + 3] << "\n";
		assert(lp::stack[env_i + yi + 3].is_variable());
		var_regs[a] = lp::stack[env_i + yi + 3].get_ptr();
		++cap;
	}

	inline void Program::put_structure(int f, int n, int vi)
	{
		lp::heap[heap_i] = term(f,n);
		var_regs[vi] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::put_list(int vi)
	{
		lp::heap[heap_i] = term::make_list();
		var_regs[vi] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::put_constant(int c, int a)
	{
		lp::heap[heap_i] = term::make_constant(c);
		var_regs[a] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::put_int(long long v, int a)
	{
		lp::heap[heap_i] = term(v);
		var_regs[a] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::put_float(double v, int a)
	{
		lp::heap[heap_i] = term(v);
		var_regs[a] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::set_variable(int v)
	{
		lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
		var_regs[v] = &lp::heap[heap_i];
		++heap_i;
		++cap;
	}

	inline void Program::set_stack_variable(int yi)
	{
		lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
		// std::cerr << "set stack variable address: " << &lp::stack[env_i + yi + 3] << "\n";
		lp::stack[env_i + yi + 3] = term::make_variable(&lp::heap[heap_i]);
		++heap_i;
		++cap;
	}

	inline void Program::set_value(int v)
	{
		lp::heap[heap_i] = term::make_variable(var_regs[v]); // REF?
		++heap_i;
		++cap;
	}

	inline void Program::set_stack_value(int yi)
	{
		// Assumes that variable has been globalized and yi already points to it
		// Find original variable by yi's reference
		//assert( lp::stack[env_i + yi + 3].is_ref() );
		lp::heap[heap_i] = term::make_variable(lp::stack[env_i + yi + 3].get_ptr());
		++heap_i;
		++cap;
	}

	inline void Program::set_constant(int c)
	{
		lp::heap[heap_i] = term::make_constant(c);
		++heap_i;
		++cap;
	}

	inline void Program::set_int(long long v)
	{
		lp::heap[heap_i] = term(v);
		++heap_i;
		++cap;
	}

	inline void Program::set_float(double v)
	{
		lp::heap[heap_i] = term(v);
		++heap_i;
		++cap;
	}

	inline void Program::get_variable(int x, int a)
	{
		// Make x point to whatever a is pointing to
		var_regs[x] = var_regs[a];
		++cap;
	}

	inline void Program::get_stack_variable(int yi, int a)
	{
		// Make yi reference whatever a is pointing to
		//std::cerr << "get_stack_variable: Y" << yi << " => " << *var_regs[a] << " (env_i: " << env_i << ")\n";
		lp::stack[env_i + yi + 3] = term::make_variable(var_regs[a]);
		++cap;
	}

	inline void Program::get_value(int s, int t)
	{
		if (unify(var_regs[s],var_regs[t])) {
			++cap;
		} else {
			backtrack();
		}
	}

	inline void Program::get_stack_value(int yi, int t)
	{
		if (unify(&lp::stack[env_i + yi + 3],var_regs[t])) {
			++cap;
		} else {
			backtrack();
		}
	}

	inline void Program::proceed()
	{
		cap = cap2;
		qlevel = qlevel2;
	}

	inline void Program::noop()
	{
		++cap;
	}

	inline void Program::fail()
	{
		backtrack();
	}

	bool consult_file(const boost::filesystem::path&, Program& kb, bool fail_free = false);


} // namespace lp

#endif


